/** @file   level.h
 * @brief   Declaration of Level - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_LEVEL_H
#define H_WWW_LEVEL_H

#include "objective.h"
#include "route.h"

namespace WeWantWar {


/** @class  Level
 * @brief   Represents a game level.
 * @author  Tomi Lamminsaari
 *
 * The Level loading is based on so called Level Description File. An instance
 * of this Level - class loads that description file and initializes the
 * structures so that we can start playing the level.
 */
class Level
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** @struct PlayerInfo
   * @brief   During the level loading the player's starting point is stored
   *          in a structure like this.
   *
   * A structure where we store the player data during the loading
   * of the map.
   */
  struct PlayerInfo {
    /** Starting X coordinate in blocks from topleft corner of the map. */
    int plaX;
    /** Starting Y coordinate in blocks from topleft corner of the map. */
    int plaY;
    /** The starting angle the player is facing. */
    int plaAngle;
    /** ID code for the Player - object. */
    int plaID;
  };

  /** Possible IO errorcodes.
   */
  enum IoError {
    /** Level Description file was ok. */
    ENone,
    /** Hash-check failed. */
    EHashCheckFailed = -1,
    /** Invalid fileformat. (no WEWANTWAR_LEVEL tag) */
    EInvalidFileFormat = -2,
    /** Failed to open the file */
    EFailedToOpen = -3,
    /** Parse error while parsing the description file. */
    EParseError = -4,
    /** Failed to load the FMP map file */
    EFmpMapFailure = -5
  };


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	Level();


	/** Destructor
   */
	virtual ~Level();

private:

	/** Copy constructor.
   * @param     rO                Reference to another Level
   */
  Level( const Level& rO );

	/** Assignment operator
   * @param     rO                Reference to another Level
   * @return    Reference to us.
   */
  Level& operator = ( const Level& rO );

public:


  ///
  /// Methods
  /// =======

  /** Loads the mapdata from given file.
   * @param     levelfile         Name of the Map Description File
   * @return    Nonzero if fails.
   */
  int load( const std::string& levelfile );
  


  ///
  /// Getter methods
  /// ==============
  
  /** Returns a pointer to an instance of Objective that represents the
   * index'th objective of this level.
   * @param     index             The Objective number we should return.
   * @return    Pointer to the Obejctive or nullpointer if there is no such
   *            objective.
   */
  Objective* getObjective( int index ) const;
  
  /** Returns the number of objectives there are.
   * @return    Number of objectives.
   */
  int objectiveCount() const;

  /** Returns the number of UNFINISHED objectives there are still left.
   * @return    Number of unfinished objectives.
   */
  int unfinishedObjectives() const;
  
  /** Returns the Route that matches with given name.
   * @param     routename         Name of the route
   * @return    The Route. If there is no route with given name, an empty
   *            route will be returned.
   */
  Route getRoute( const std::string& routename ) const;
  
  

  ///
  /// Public members
  /// ==============
  
  /** This vector contains the level story. Each line is its own element. */
  std::vector< std::string > storylines;
  
  /** Name of the titlepicture file. This is the picture we use as story
   * background.
   */
  std::string titlepicfile;
  
  /** Name of the story bitmap. This bitmap contains the story text that
   * scrolls below the titlepic.
   */
  std::string storypicfile;
  
  /** Name of the background musicfile. */
  std::vector< std::string > musicfile;
  
  /** Name of the FMP mapfile. */
  std::string mapfile;
  
  /** The weather. This integer has bitflags that tells wich weather effects
   * are on. See @c Weather - class for more information.
   */
  int weather;
  
  /** The starting position of the Player. */
  PlayerInfo playerStartingPos;
  
  /** This vector holds the Routes loaded from the description file. */
  std::vector< Route >  routeTable;
  
  
  /** The original timebonus */
  int timebonus;
  
  /** The leveltimer that counts how long we've been playing this level. */
  int levelTimer;

  /** Flag that indicates if night mode is on. */
  bool nightModeOn;
  
protected:

  ///
  /// Protected method
  /// ================
  
  /** Releases all the resources allocated for this level.
   */
  void cleanup();
  
  /** Parses the Map Description File.
   * @param     rIn               The input stream
   * @return    Nonzero if errors were encountered.
   */
  int parseLevelFile( std::istream& rIn );
  
  /** Reads the contents of general information - tag.
   * @param     rIn               The input stream.
   * @return    Nonzero if fails.
   */
  int readGeneralInformation( std::istream& rIn );
  
  /** Reads the player information
   * @param     rIn               The input stream
   * @return    Nonzero if fails
   */
  int readPlayerInformation( std::istream& rIn );
  
  /** Reads the level story.
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  int readStory( std::istream& rIn );
  
  /** Includes another file.
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  int includeFile( std::istream& rIn );
  
  
  
  ///
  /// Members
  /// =======

  /** This vector contains the objectives this level has. */
  std::vector< Objective* > m_objectives;
  
  /** The objective number the player is currently doing. */
  int m_activeObjective;
  
  
  
private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: level.h,v $
 * Revision 1.3  2006/06/17 21:44:30  lamminsa
 * nightModeOn member added.
 *
 * Revision 1.2  2006/05/15 19:20:01  lamminsa
 * Level can have more than one mp3 file.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.1  2005-11-23 00:27:28+02  lamminsa
 * IoError codes
 *
 * Revision 1.0  2005-11-06 01:17:00+02  lamminsa
 * Initial revision
 *
 */
 
